import requests
import os
import time
from enum import Enum

# Azure configuration
AZURE_REGION = "AZURE_REGION"
AZURE_SUBSCRIPTION_KEY = "AZURE_SUBSCRIPTION_KEY"

# Azure Speech-to-Text endpoints
STT_BASE_URL = f"https://{AZURE_REGION}.api.cognitive.microsoft.com/speechtotext/v3.2"
STT_TOKEN_ENDPOINT = f"https://{AZURE_REGION}.api.cognitive.microsoft.com/sts/v1.0/issuetoken"

PROGRESS_UPLOADED = 40  # % of overall progress
PROGRESS_START_PROCESSING = 45  # % of overall progress

def get_info():
    return {
        "auto-lang": False,
        "languages": [
            "af-ZA", "am-ET", "ar-AE", "ar-BH", "ar-DZ", "ar-EG", "ar-IQ", "ar-JO",
            "ar-KW", "ar-LB", "ar-LY", "ar-MA", "ar-OM", "ar-QA", "ar-SA", "ar-SY", 
            "ar-TN", "ar-YE", "as-IN", "az-AZ", "bg-BG", "bn-BD", "bn-IN", "bs-BA", 
            "ca-ES", "cs-CZ", "cy-GB", "da-DK", "de-AT", "de-CH", "de-DE", "el-GR", 
            "en-AU", "en-CA", "en-GB", "en-HK", "en-IE", "en-IN", "en-KE", "en-NG", 
            "en-NZ", "en-PH", "en-SG", "en-TZ", "en-US", "en-ZA", "es-AR", "es-BO", 
            "es-CL", "es-CO", "es-CR", "es-CU", "es-DO", "es-EC", "es-ES", "es-GQ", 
            "es-GT", "es-HN", "es-MX", "es-NI", "es-PA", "es-PE", "es-PR", "es-PY", 
            "es-SV", "es-US", "es-UY", "es-VE", "et-EE", "eu-ES", "fa-IR", "fi-FI", 
            "fil-PH", "fr-BE", "fr-CA", "fr-CH", "fr-FR", "ga-IE", "gl-ES", "gu-IN", 
            "he-IL", "hi-IN", "hr-HR", "hu-HU", "hy-AM", "id-ID", "is-IS", "it-IT", 
            "ja-JP", "jv-ID", "ka-GE", "kk-KZ", "km-KH", "kn-IN", "ko-KR", "lo-LA", 
            "lt-LT", "lv-LV", "mk-MK", "ml-IN", "mn-MN", "mr-IN", "ms-MY", "mt-MT", 
            "my-MM", "nb-NO", "ne-NP", "nl-BE", "nl-NL", "or-IN", "pa-IN", "pl-PL", 
            "ps-AF", "pt-BR", "pt-PT", "ro-RO", "ru-RU", "si-LK", "sk-SK", "sl-SI", 
            "so-SO", "sq-AL", "sr-RS", "su-ID", "sv-SE", "sw-KE", "sw-TZ", "ta-IN", 
            "ta-LK", "ta-MY", "ta-SG", "te-IN", "th-TH", "tr-TR", "uk-UA", "ur-IN", 
            "ur-PK", "uz-UZ", "vi-VN", "zh-CN", "zh-HK", "zh-TW", "zu-ZA"
        ]
    }

def speech_to_text(audio_path, language_codes, progress_callback=None):
    """
    Convert speech to text using Azure Batch Transcription API
    
    Args:
        audio_path: Path to audio file
        language_codes: List of language codes (e.g., ["vi-VN", "en-US"])
        progress_callback: Callback function for progress updates
    
    Returns:
        Dictionary with transcript data
    """
    def upload_progress_handler(upload_progress, message=""):
        if progress_callback:
            return progress_callback(int(upload_progress * PROGRESS_UPLOADED / 100), message)
        return True
    
    # Upload file to Azure
    content_url = upload_to_azure(audio_path, upload_progress_handler)
    if not content_url:
        return None
    
    # Get authentication token
    auth_token = get_azure_token()
    
    # Determine endpoint
    transcription_endpoint = f"{STT_BASE_URL}/transcriptions"
    if is_special_azure_region(AZURE_REGION):
        transcription_endpoint = generate_azure_stt_endpoint(AZURE_REGION, AzureSTTEndpointMode.TRANSCRIPTION)
    
    # Create transcription request
    headers = {
        "Authorization": f"Bearer {auth_token}",
        "Content-Type": "application/json"
    }
    
    # Use first language as primary, rest as alternatives
    primary_language = language_codes[0] if language_codes else "en-US"
    
    payload = {
        "contentUrls": [content_url],
        "locale": primary_language,
        "displayName": f"Transcription_{int(time.time())}",
        "properties": {
            "wordLevelTimestampsEnabled": True,
            "punctuationMode": "DictatedAndAutomatic",
            "profanityFilterMode": "None"
        }
    }
    
    # Add language identification if multiple languages
    if len(language_codes) > 1:
        payload["properties"]["languageIdentification"] = {
            "candidateLocales": language_codes
        }
    
    response = requests.post(transcription_endpoint, headers=headers, json=payload)
    
    if response.status_code not in (200, 201):
        handle_response_error(response, "Failed to start batch transcription")
    
    transcription_data = response.json()
    transcription_url = transcription_data["self"]
    
    if progress_callback and not progress_callback(PROGRESS_START_PROCESSING):
        # Cancel transcription if user cancels
        delete_transcription(transcription_url, auth_token)
        return None
    
    def transcription_progress_handler(server_progress, message=""):
        if progress_callback:
            return progress_callback(
                PROGRESS_START_PROCESSING + int(server_progress * (1 - PROGRESS_START_PROCESSING/100)), 
                message
            )
        return True
    
    # Poll for completion
    result_data = poll_transcription(transcription_url, auth_token, transcription_progress_handler)
    if not result_data:
        return None
    
    # Process results
    transcript = process_transcription_results(result_data)
    
    # Cleanup: delete transcription from Azure
    delete_transcription(transcription_url, auth_token)
    
    if progress_callback:
        progress_callback(100)
    
    return transcript

def upload_to_azure(file_path, progress_callback=None):
    """
    Upload audio file to accessible URL for Azure
    Note: Azure Batch Transcription requires a publicly accessible URL
    You'll need to implement your own storage solution (Azure Blob, AWS S3, etc.)
    """
    # This is a placeholder - you need to implement actual file upload
    # to Azure Blob Storage or another accessible location
    
    # Example using Azure Blob Storage:
    # 1. Upload to Azure Blob Storage
    # 2. Generate SAS URL with read permissions
    # 3. Return the SAS URL
    
    raise NotImplementedError(
        "You need to implement file upload to Azure Blob Storage or another publicly accessible location. "
        "Azure Batch Transcription API requires a URL to access the audio file."
    )

def poll_transcription(transcription_url, auth_token, progress_callback=None):
    """Poll transcription status until complete"""
    headers = {
        "Authorization": f"Bearer {auth_token}"
    }
    
    poll_interval = 5  # seconds
    progress = 0
    
    while True:
        response = requests.get(transcription_url, headers=headers)
        
        if response.status_code == 429:
            poll_interval = min(2 * poll_interval, 60)
            time.sleep(poll_interval)
            continue
        elif response.status_code != 200:
            handle_response_error(response, "Failed to get transcription status")
        
        data = response.json()
        status = data.get("status")
        
        if status == "Succeeded":
            # Get transcription results
            files_url = data["links"]["files"]
            return get_transcription_results(files_url, auth_token)
        elif status == "Failed":
            error_msg = data.get("properties", {}).get("error", {}).get("message", "Transcription failed")
            raise Exception(error_msg)
        
        # Update progress
        if progress_callback:
            # Azure doesn't provide progress percentage, so estimate
            progress = min(progress + 5, 95)
            if not progress_callback(progress):
                delete_transcription(transcription_url, auth_token)
                return None
        
        time.sleep(poll_interval)

def get_transcription_results(files_url, auth_token):
    """Get transcription result files"""
    headers = {
        "Authorization": f"Bearer {auth_token}"
    }
    
    response = requests.get(files_url, headers=headers)
    if response.status_code != 200:
        handle_response_error(response, "Failed to get transcription files")
    
    files = response.json()["values"]
    
    # Find the transcription result file
    result_file = None
    for file in files:
        if file["kind"] == "Transcription":
            result_file = file
            break
    
    if not result_file:
        raise Exception("Transcription result file not found")
    
    # Download the result
    content_url = result_file["links"]["contentUrl"]
    response = requests.get(content_url)
    if response.status_code != 200:
        handle_response_error(response, "Failed to download transcription results")
    
    return response.json()

def process_transcription_results(result_data):
    """Process Azure transcription results into standard format"""
    transcript = {
        "duration": 0,
        "segments": []
    }
    
    combined_phrases = result_data.get("combinedRecognizedPhrases", [])
    if combined_phrases:
        # Get total duration from last phrase
        last_phrase = combined_phrases[-1]
        duration_ticks = last_phrase.get("offsetInTicks", 0) + last_phrase.get("durationInTicks", 0)
        transcript["duration"] = ticks_to_ms(duration_ticks)
    
    recognized_phrases = result_data.get("recognizedPhrases", [])
    
    for phrase in recognized_phrases:
        best = phrase.get("nBest", [{}])[0]
        
        segment = {
            "start": ticks_to_ms(phrase.get("offsetInTicks", 0)),
            "end": ticks_to_ms(phrase.get("offsetInTicks", 0) + phrase.get("durationInTicks", 0)),
            "text": best.get("display", ""),
            "words": []
        }
        
        # Process words with timestamps
        words = best.get("words", [])
        for word_data in words:
            word = {
                "text": word_data.get("word", ""),
                "start": ticks_to_ms(word_data.get("offsetInTicks", 0)),
                "end": ticks_to_ms(word_data.get("offsetInTicks", 0) + word_data.get("durationInTicks", 0))
            }
            segment["words"].append(word)
        
        transcript["segments"].append(segment)
    
    return transcript

def delete_transcription(transcription_url, auth_token):
    """Delete transcription from Azure to cleanup"""
    headers = {
        "Authorization": f"Bearer {auth_token}"
    }
    try:
        requests.delete(transcription_url, headers=headers)
    except Exception:
        pass  # Ignore errors during cleanup

def get_azure_token():
    """Get Azure authentication token"""
    endpoint = STT_TOKEN_ENDPOINT
    if is_special_azure_region(AZURE_REGION):
        endpoint = generate_azure_stt_endpoint(AZURE_REGION, AzureSTTEndpointMode.TOKEN)
    
    headers = {
        "Ocp-Apim-Subscription-Key": AZURE_SUBSCRIPTION_KEY
    }
    
    response = requests.post(endpoint, headers=headers)
    if response.status_code != 200:
        handle_response_error(response, "Error fetching token")
    
    return response.text

def ticks_to_ms(ticks):
    """Convert Azure ticks (100-nanosecond intervals) to milliseconds"""
    return int(ticks / 10000)

class AzureSTTEndpointMode(Enum):
    TOKEN = "token"
    TRANSCRIPTION = "transcription"

def generate_azure_stt_endpoint(region, mode):
    """Generate Azure STT endpoint URL for special regions"""
    try:
        if not region:
            raise ValueError("Region cannot be empty")
        
        base_domain = "microsoft.com"
        
        if region.startswith("china"):
            base_domain = "azure.cn"
        elif region.startswith("usgov"):
            base_domain = "microsoft.us" if mode == AzureSTTEndpointMode.TOKEN else "azure.us"
        elif region.startswith("usdod"):
            base_domain = "microsoft.us"
        
        if mode == AzureSTTEndpointMode.TOKEN:
            return f"https://{region}.api.cognitive.{base_domain}/sts/v1.0/issuetoken"
        elif mode == AzureSTTEndpointMode.TRANSCRIPTION:
            return f"https://{region}.api.cognitive.{base_domain}/speechtotext/v3.2/transcriptions"
        else:
            raise ValueError(f"Unsupported endpoint mode: {mode}")
            
    except Exception as e:
        raise Exception(f"Failed to generate Azure STT endpoint: {str(e)}")

def is_special_azure_region(region):
    """Check if region requires special endpoint handling"""
    return (region.startswith("china") or 
            region.startswith("usgov") or 
            region.startswith("usdod"))

def handle_response_error(response, log_msg):
    """Handle API response errors"""
    if log_msg:
        log_message(log_msg)
    
    error_msg = ""
    try:
        response_json = response.json()
        error_msg = response_json.get("error", {}).get("message")
        if not error_msg:
            error_msg = response_json.get("message")
        if not error_msg:
            error_msg = str(response_json.get("error"))
    except Exception:
        pass
    
    if error_msg:
        log_message(f"Error {response.status_code} - {error_msg}")
        raise Exception(error_msg)
    else:
        raise Exception(f"Error {response.status_code} - {response.text}")

def log_message(message):
    """Save message to the app log file if possible, otherwise print it"""
    log = getattr(globals().get('app', None), 'Log', print)
    log(message)

def raise_error(message):
    """Raise an exception. Execution will stop"""
    raise Exception(message)